<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;

use App\Http\Services\DBClientConnection;

use App\Models\clientes\InstrumentosRiasAPNIve;
use App\Models\clientes\HistoricoInsRiasAPNIve;

class InstrumentosRiasAPNIveController extends Controller
{
    private $cliente;

    function __construct() {
        // Crear la conexion temporal al esquema del cliente
        $this->middleware(function ($request, $next) {
            if (Auth::user() !== null) {
                $this->cliente = Auth::user()->cli_fk_id;
                $connection = new DBClientConnection();
                config(['database.connections.'.$this->cliente => $connection->getConnectionArray($this->cliente)]);
            }

            return $next($request);
        });
    }

    public function obtenerInstrumentos(Request $request) {
        return InstrumentosRiasAPNIve::on($this->cliente)
                                    ->orderBy('iri_activa', 'desc')
                                    ->orderBy('iri_pk_id', 'desc')
                                    ->get()->toArray();
    }

    public function crearInstrumento(Request $request) {
        $id = null;
        $nuevaVersion = null;

        DB::connection($this->cliente)->transaction(function() use($request, &$id, &$nuevaVersion) {
            $nuevaVersion = DB::connection($this->cliente)->select(
                'select nextval(\'"'.$this->cliente.'".seq_rias_apn_ive\')'
            )[0]->nextval;

            $usuario = Auth::user()->id;

            $id = InstrumentosRiasAPNIve::on($this->cliente)->create([
                'iri_version' => $nuevaVersion,
                'iri_comentarios' => $request->iri_comentarios,
                'iri_instrumento' => json_encode($request->iri_instrumento),
                'iri_usu_creacion' => $usuario
            ])->iri_pk_id;

            HistoricoInsRiasAPNIve::on($this->cliente)->create([
                'iri_fk_id' => $id,
                'usu_fk_id' => $usuario,
                'hri_accion' => 'Creación de una nueva versión del instrumento'
            ]);
        });

        return [$id, $nuevaVersion];
    }

    public function actualizarInstrumento(Request $request, $id) {
        DB::connection($this->cliente)->transaction(function() use($request, &$id) {
            $usuario = Auth::user()->id;

            InstrumentosRiasAPNIve::on($this->cliente)->where('iri_pk_id', $id)->update([
                'iri_comentarios' => $request->iri_comentarios,
                'iri_instrumento' => $request->iri_instrumento
            ]);

            HistoricoInsRiasAPNIve::on($this->cliente)->create([
                'iri_fk_id' => $id,
                'usu_fk_id' => $usuario,
                'hri_accion' => 'Actualización del instrumento'
            ]);
        });
    }

    public function activarInstrumento(Request $request, $id) {
        DB::connection($this->cliente)->transaction(function() use($request, &$id) {
            $usuario = Auth::user()->id;

            $idActivo = InstrumentosRiasAPNIve::on($this->cliente)
                                            ->where('iri_activa', true)
                                            ->pluck('iri_pk_id')[0];

            // Inactivar el actual
            InstrumentosRiasAPNIve::on($this->cliente)->where('iri_pk_id', $idActivo)->update([
                'iri_activa' => false
            ]);

            HistoricoInsRiasAPNIve::on($this->cliente)->create([
                'iri_fk_id' => $idActivo,
                'usu_fk_id' => $usuario,
                'hri_accion' => 'El instrumento pasa a estado inactivo'
            ]);

            // Activar el nuevo instrumento
            InstrumentosRiasAPNIve::on($this->cliente)->where('iri_pk_id', $id)->update([
                'iri_activa' => true
            ]);

            HistoricoInsRiasAPNIve::on($this->cliente)->create([
                'iri_fk_id' => $id,
                'usu_fk_id' => $usuario,
                'hri_accion' => 'El instrumento pasa a estado activo'
            ]);
        });
    }

    public function obtenerHistoricoVersion(Request $request, $id) {
        return HistoricoInsRiasAPNIve::on($this->cliente)
                                   ->selectRaw('historico_ins_rias_apn_ive.*, users.name')
                                   ->join('base.users', 'id', 'usu_fk_id')
                                   ->where('iri_fk_id', $id)
                                   ->orderBy('created_at', 'desc')
                                   ->get()->toArray();
    }
}
