<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;

use App\Models\clientes\EstadoConsolidacionIndicadores;

use App\Http\Services\DBClientConnection;

class IndicadoresAuditoriaPamecController extends Controller
{
    private $cliente;

    function __construct() {
        // Crear la conexion temporal al esquema del cliente
        $this->middleware(function ($request, $next) {
            if (Auth::user() !== null) {
                $this->cliente = Auth::user()->cli_fk_id;
                $connection = new DBClientConnection();
                config(['database.connections.'.$this->cliente => $connection->getConnectionArray($this->cliente)]);
            }

            return $next($request);
        });
    }

    public function establecerCliente($cliente) {
        $this->cliente = $cliente;
    }

    public function obtenerIndicadores(Request $request, $ano, $mes, $municipio, $prestador, $visita) {
        $parametros = [];
        $parametrosAplicados = 0;
        $where = "";

        // Ano
        if (intval($ano) !== -1) {
            $where .= "@prefijo@_ano = ?";
            array_push($parametros, $ano);
            $parametrosAplicados++;
        }

        // Mes
        if (intval($mes) !== -1) {
            if ($where !== "") { $where .= " and "; }
            $where .= "@prefijo@_mes = ?";
            array_push($parametros, $mes);
            $parametrosAplicados++;
        }

        // Municipio
        if ($where !== "") { $where .= " and "; }
        $where .= "mun_fk_id = ?";
        array_push($parametros, $municipio);
        $parametrosAplicados++;

        // Prestador
        if ($where !== "") { $where .= " and "; }
        $where .= "rpr_fk_id = ?";
        array_push($parametros, $prestador);
        $parametrosAplicados++;

        // Visita
        if ($where !== "") { $where .= " and "; }
        $where .= "plv_fk_id = ?";
        array_push($parametros, $visita);
        $parametrosAplicados++;

        // Replicar los parametros para todas las consultas
        for ($i = 0; $i < 2; $i++) {
            for ($j = 0; $j < $parametrosAplicados; $j++) {
                array_push($parametros, $parametros[$j]);
            }
        }

        return DB::connection($this->cliente)->select("
            select ipg.*, ipa.*, ipo.*
            from (
                select coalesce(sum(ipg_visitas_programadas), 0) as ipg_visitas_programadas,
                       coalesce(sum(ipg_visitas_por_realizar), 0) as ipg_visitas_por_realizar,
                       coalesce(sum(ipg_visitas_por_realizar)::numeric / nullif(sum(ipg_visitas_programadas)::numeric, 0), 0) * 100 as ipg_prc_visitas_por_realizar,
                       coalesce(sum(ipg_visitas_realizadas), 0) as ipg_visitas_realizadas,
                       coalesce(sum(ipg_visitas_realizadas)::numeric / nullif(sum(ipg_visitas_programadas)::numeric, 0), 0) * 100 as ipg_prc_visitas_realizadas,
                       coalesce(sum(ipg_visitas_efectivas), 0) as ipg_visitas_efectivas,
                       coalesce(sum(ipg_visitas_efectivas)::numeric / nullif(sum(ipg_visitas_realizadas)::numeric, 0), 0) * 100 as ipg_prc_visitas_efectivas,
                       coalesce(sum(ipg_visitas_no_efectivas), 0) as ipg_visitas_no_efectivas,
                       coalesce(sum(ipg_visitas_no_efectivas)::numeric / nullif(sum(ipg_visitas_realizadas)::numeric, 0), 0) * 100 as ipg_prc_visitas_no_efectivas,

                       coalesce(sum(ipg_total_criterios), 0) as ipg_total_criterios,
                       coalesce(sum(ipg_total_criterios_cumplen), 0) as ipg_total_criterios_cumplen,
                       coalesce(sum(ipg_total_criterios_no_cumplen), 0) as ipg_total_criterios_no_cumplen,
                       coalesce(sum(ipg_total_criterios_cumplen)::numeric / nullif(sum(ipg_total_criterios)::numeric, 0), 0) * 100 as ipg_prc_cumplimiento,
                       coalesce(sum(ipg_visitas_ejec_baja), 0) as ipg_visitas_ejec_baja,
                       coalesce(sum(ipg_visitas_ejec_media), 0) as ipg_visitas_ejec_media,
                       coalesce(sum(ipg_visitas_ejec_alta), 0) as ipg_visitas_ejec_alta,

                       coalesce(sum(ipg_evaluacion_1_cumple), 0) as ipg_evaluacion_1_cumple,
                       coalesce(sum(ipg_evaluacion_1_cumple)::numeric / nullif(sum(ipg_visitas_realizadas_completadas)::numeric, 0), 0) * 100 as ipg_prc_evaluacion_1_cumple,
                       coalesce(sum(ipg_evaluacion_2_cumple), 0) as ipg_evaluacion_2_cumple,
                       coalesce(sum(ipg_evaluacion_2_cumple)::numeric / nullif(sum(ipg_visitas_realizadas_completadas)::numeric, 0), 0) * 100 as ipg_prc_evaluacion_2_cumple,
                       coalesce(sum(ipg_evaluacion_3_cumple), 0) as ipg_evaluacion_3_cumple,
                       coalesce(sum(ipg_evaluacion_3_cumple)::numeric / nullif(sum(ipg_visitas_realizadas_completadas)::numeric, 0), 0) * 100 as ipg_prc_evaluacion_3_cumple,
                       coalesce(sum(ipg_evaluacion_4_cumple), 0) as ipg_evaluacion_4_cumple,
                       coalesce(sum(ipg_evaluacion_4_cumple)::numeric / nullif(sum(ipg_visitas_realizadas_completadas)::numeric, 0), 0) * 100 as ipg_prc_evaluacion_4_cumple,
                       coalesce(sum(ipg_evaluacion_5_cumple), 0) as ipg_evaluacion_5_cumple,
                       coalesce(sum(ipg_evaluacion_5_cumple)::numeric / nullif(sum(ipg_visitas_realizadas_completadas)::numeric, 0), 0) * 100 as ipg_prc_evaluacion_5_cumple,
                       coalesce(sum(ipg_evaluacion_6_cumple), 0) as ipg_evaluacion_6_cumple,
                       coalesce(sum(ipg_evaluacion_6_cumple)::numeric / nullif(sum(ipg_visitas_realizadas_completadas)::numeric, 0), 0) * 100 as ipg_prc_evaluacion_6_cumple,
                       coalesce(sum(ipg_evaluacion_7_cumple), 0) as ipg_evaluacion_7_cumple,
                       coalesce(sum(ipg_evaluacion_7_cumple)::numeric / nullif(sum(ipg_visitas_realizadas_completadas)::numeric, 0), 0) * 100 as ipg_prc_evaluacion_7_cumple,
                       coalesce(sum(ipg_evaluacion_8_cumple), 0) as ipg_evaluacion_8_cumple,
                       coalesce(sum(ipg_evaluacion_8_cumple)::numeric / nullif(sum(ipg_visitas_realizadas_completadas)::numeric, 0), 0) * 100 as ipg_prc_evaluacion_8_cumple,
                       coalesce(sum(ipg_evaluacion_9_cumple), 0) as ipg_evaluacion_9_cumple,
                       coalesce(sum(ipg_evaluacion_9_cumple)::numeric / nullif(sum(ipg_visitas_realizadas_completadas)::numeric, 0), 0) * 100 as ipg_prc_evaluacion_9_cumple,
                       coalesce(sum(ipg_evaluacion_10_cumple), 0) as ipg_evaluacion_10_cumple,
                       coalesce(sum(ipg_evaluacion_10_cumple)::numeric / nullif(sum(ipg_visitas_realizadas_completadas)::numeric, 0), 0) * 100 as ipg_prc_evaluacion_10_cumple
                from \"".$this->cliente."\".indicadores_pamec_global
                where ".str_replace("@prefijo@", "ipg", $where)."
            ) as ipg,
            (
                select coalesce(sum(ipa_pamec_acreditacion), 0) as ipa_pamec_acreditacion,
                       coalesce(sum(ipa_acred_ejec_baja), 0) as ipa_acred_ejec_baja,
                       coalesce(sum(ipa_acred_ejec_media), 0) as ipa_acred_ejec_media,
                       coalesce(sum(ipa_acred_ejec_alta), 0) as ipa_acred_ejec_alta,
                       coalesce(sum(ipa_acred_evaluacion_1_cumple)::numeric / nullif(sum(ipa_pamec_acreditacion)::numeric, 0), 0) * 100 as ipa_acred_evaluacion_1_cumple,
                       coalesce(sum(ipa_acred_evaluacion_2_cumple)::numeric / nullif(sum(ipa_pamec_acreditacion)::numeric, 0), 0) * 100 as ipa_acred_evaluacion_2_cumple,
                       coalesce(sum(ipa_acred_evaluacion_3_cumple)::numeric / nullif(sum(ipa_pamec_acreditacion)::numeric, 0), 0) * 100 as ipa_acred_evaluacion_3_cumple,
                       coalesce(sum(ipa_acred_evaluacion_4_cumple)::numeric / nullif(sum(ipa_pamec_acreditacion)::numeric, 0), 0) * 100 as ipa_acred_evaluacion_4_cumple,
                       coalesce(sum(ipa_acred_evaluacion_5_cumple)::numeric / nullif(sum(ipa_pamec_acreditacion)::numeric, 0), 0) * 100 as ipa_acred_evaluacion_5_cumple,
                       coalesce(sum(ipa_acred_evaluacion_6_cumple)::numeric / nullif(sum(ipa_pamec_acreditacion)::numeric, 0), 0) * 100 as ipa_acred_evaluacion_6_cumple,
                       coalesce(sum(ipa_acred_evaluacion_7_cumple)::numeric / nullif(sum(ipa_pamec_acreditacion)::numeric, 0), 0) * 100 as ipa_acred_evaluacion_7_cumple,
                       coalesce(sum(ipa_acred_evaluacion_8_cumple)::numeric / nullif(sum(ipa_pamec_acreditacion)::numeric, 0), 0) * 100 as ipa_acred_evaluacion_8_cumple,
                       coalesce(sum(ipa_acred_evaluacion_9_cumple)::numeric / nullif(sum(ipa_pamec_acreditacion)::numeric, 0), 0) * 100 as ipa_acred_evaluacion_9_cumple,
                       coalesce(sum(ipa_acred_evaluacion_10_cumple)::numeric / nullif(sum(ipa_pamec_acreditacion)::numeric, 0), 0) * 100 as ipa_acred_evaluacion_10_cumple
                from \"".$this->cliente."\".indicadores_pamec_acreditacion
                where ".str_replace("@prefijo@", "ipa", $where)."
            ) as ipa,
            (
                select coalesce(sum(ipo_pamec_si_calidad), 0) as ipo_pamec_si_calidad,
                       coalesce(sum(ipo_si_calidad_ejec_baja), 0) as ipo_si_calidad_ejec_baja,
                       coalesce(sum(ipo_si_calidad_ejec_media), 0) as ipo_si_calidad_ejec_media,
                       coalesce(sum(ipo_si_calidad_ejec_alta), 0) as ipo_si_calidad_ejec_alta,
                       coalesce(sum(ipo_si_cal_evaluacion_1_cumple)::numeric / nullif(sum(ipo_pamec_si_calidad)::numeric, 0), 0) * 100 as ipo_si_cal_evaluacion_1_cumple,
                       coalesce(sum(ipo_si_cal_evaluacion_2_cumple)::numeric / nullif(sum(ipo_pamec_si_calidad)::numeric, 0), 0) * 100 as ipo_si_cal_evaluacion_2_cumple,
                       coalesce(sum(ipo_si_cal_evaluacion_3_cumple)::numeric / nullif(sum(ipo_pamec_si_calidad)::numeric, 0), 0) * 100 as ipo_si_cal_evaluacion_3_cumple,
                       coalesce(sum(ipo_si_cal_evaluacion_4_cumple)::numeric / nullif(sum(ipo_pamec_si_calidad)::numeric, 0), 0) * 100 as ipo_si_cal_evaluacion_4_cumple,
                       coalesce(sum(ipo_si_cal_evaluacion_5_cumple)::numeric / nullif(sum(ipo_pamec_si_calidad)::numeric, 0), 0) * 100 as ipo_si_cal_evaluacion_5_cumple,
                       coalesce(sum(ipo_si_cal_evaluacion_6_cumple)::numeric / nullif(sum(ipo_pamec_si_calidad)::numeric, 0), 0) * 100 as ipo_si_cal_evaluacion_6_cumple,
                       coalesce(sum(ipo_si_cal_evaluacion_7_cumple)::numeric / nullif(sum(ipo_pamec_si_calidad)::numeric, 0), 0) * 100 as ipo_si_cal_evaluacion_7_cumple,
                       coalesce(sum(ipo_si_cal_evaluacion_8_cumple)::numeric / nullif(sum(ipo_pamec_si_calidad)::numeric, 0), 0) * 100 as ipo_si_cal_evaluacion_8_cumple,
                       coalesce(sum(ipo_si_cal_evaluacion_9_cumple)::numeric / nullif(sum(ipo_pamec_si_calidad)::numeric, 0), 0) * 100 as ipo_si_cal_evaluacion_9_cumple,
                       coalesce(sum(ipo_si_cal_evaluacion_10_cumple)::numeric / nullif(sum(ipo_pamec_si_calidad)::numeric, 0), 0) * 100 as ipo_si_cal_evaluacion_10_cumple,

                       coalesce(sum(ipo_pamec_gr_procesos), 0) as ipo_pamec_gr_procesos,
                       coalesce(sum(ipo_gr_proc_ejec_baja), 0) as ipo_gr_proc_ejec_baja,
                       coalesce(sum(ipo_gr_proc_ejec_media), 0) as ipo_gr_proc_ejec_media,
                       coalesce(sum(ipo_gr_proc_ejec_alta), 0) as ipo_gr_proc_ejec_alta,
                       coalesce(sum(ipo_gr_proc_evaluacion_1_cumple)::numeric / nullif(sum(ipo_pamec_gr_procesos)::numeric, 0), 0) * 100 as ipo_gr_proc_evaluacion_1_cumple,
                       coalesce(sum(ipo_gr_proc_evaluacion_2_cumple)::numeric / nullif(sum(ipo_pamec_gr_procesos)::numeric, 0), 0) * 100 as ipo_gr_proc_evaluacion_2_cumple,
                       coalesce(sum(ipo_gr_proc_evaluacion_3_cumple)::numeric / nullif(sum(ipo_pamec_gr_procesos)::numeric, 0), 0) * 100 as ipo_gr_proc_evaluacion_3_cumple,
                       coalesce(sum(ipo_gr_proc_evaluacion_4_cumple)::numeric / nullif(sum(ipo_pamec_gr_procesos)::numeric, 0), 0) * 100 as ipo_gr_proc_evaluacion_4_cumple,
                       coalesce(sum(ipo_gr_proc_evaluacion_5_cumple)::numeric / nullif(sum(ipo_pamec_gr_procesos)::numeric, 0), 0) * 100 as ipo_gr_proc_evaluacion_5_cumple,
                       coalesce(sum(ipo_gr_proc_evaluacion_6_cumple)::numeric / nullif(sum(ipo_pamec_gr_procesos)::numeric, 0), 0) * 100 as ipo_gr_proc_evaluacion_6_cumple,
                       coalesce(sum(ipo_gr_proc_evaluacion_7_cumple)::numeric / nullif(sum(ipo_pamec_gr_procesos)::numeric, 0), 0) * 100 as ipo_gr_proc_evaluacion_7_cumple,
                       coalesce(sum(ipo_gr_proc_evaluacion_8_cumple)::numeric / nullif(sum(ipo_pamec_gr_procesos)::numeric, 0), 0) * 100 as ipo_gr_proc_evaluacion_8_cumple,
                       coalesce(sum(ipo_gr_proc_evaluacion_9_cumple)::numeric / nullif(sum(ipo_pamec_gr_procesos)::numeric, 0), 0) * 100 as ipo_gr_proc_evaluacion_9_cumple,
                       coalesce(sum(ipo_gr_proc_evaluacion_10_cumple)::numeric / nullif(sum(ipo_pamec_gr_procesos)::numeric, 0), 0) * 100 as ipo_gr_proc_evaluacion_10_cumple,

                       coalesce(sum(ipo_pamec_seg_paciente), 0) as ipo_pamec_seg_paciente,
                       coalesce(sum(ipo_seg_pac_ejec_baja), 0) as ipo_seg_pac_ejec_baja,
                       coalesce(sum(ipo_seg_pac_ejec_media), 0) as ipo_seg_pac_ejec_media,
                       coalesce(sum(ipo_seg_pac_ejec_alta), 0) as ipo_seg_pac_ejec_alta,
                       coalesce(sum(ipo_seg_pac_evaluacion_1_cumple)::numeric / nullif(sum(ipo_pamec_seg_paciente)::numeric, 0), 0) * 100 as ipo_seg_pac_evaluacion_1_cumple,
                       coalesce(sum(ipo_seg_pac_evaluacion_2_cumple)::numeric / nullif(sum(ipo_pamec_seg_paciente)::numeric, 0), 0) * 100 as ipo_seg_pac_evaluacion_2_cumple,
                       coalesce(sum(ipo_seg_pac_evaluacion_3_cumple)::numeric / nullif(sum(ipo_pamec_seg_paciente)::numeric, 0), 0) * 100 as ipo_seg_pac_evaluacion_3_cumple,
                       coalesce(sum(ipo_seg_pac_evaluacion_4_cumple)::numeric / nullif(sum(ipo_pamec_seg_paciente)::numeric, 0), 0) * 100 as ipo_seg_pac_evaluacion_4_cumple,
                       coalesce(sum(ipo_seg_pac_evaluacion_5_cumple)::numeric / nullif(sum(ipo_pamec_seg_paciente)::numeric, 0), 0) * 100 as ipo_seg_pac_evaluacion_5_cumple,
                       coalesce(sum(ipo_seg_pac_evaluacion_6_cumple)::numeric / nullif(sum(ipo_pamec_seg_paciente)::numeric, 0), 0) * 100 as ipo_seg_pac_evaluacion_6_cumple,
                       coalesce(sum(ipo_seg_pac_evaluacion_7_cumple)::numeric / nullif(sum(ipo_pamec_seg_paciente)::numeric, 0), 0) * 100 as ipo_seg_pac_evaluacion_7_cumple,
                       coalesce(sum(ipo_seg_pac_evaluacion_8_cumple)::numeric / nullif(sum(ipo_pamec_seg_paciente)::numeric, 0), 0) * 100 as ipo_seg_pac_evaluacion_8_cumple,
                       coalesce(sum(ipo_seg_pac_evaluacion_9_cumple)::numeric / nullif(sum(ipo_pamec_seg_paciente)::numeric, 0), 0) * 100 as ipo_seg_pac_evaluacion_9_cumple,
                       coalesce(sum(ipo_seg_pac_evaluacion_10_cumple)::numeric / nullif(sum(ipo_pamec_seg_paciente)::numeric, 0), 0) * 100 as ipo_seg_pac_evaluacion_10_cumple
                from \"".$this->cliente."\".indicadores_pamec_otros_enfoques
                where ".str_replace("@prefijo@", "ipo", $where)."
            ) as ipo
        ", $parametros);
    }

    public function actualizarIndicadores(Request $request) {
        DB::connection($this->cliente)->transaction(function() use($request) {
            $ano = $request->ano;
            $mes = $request->mes;

            $this->actualizarIndicadoresGlobales($ano, $mes);
            $this->actualizarIndicadoresAcreditacion($ano, $mes);
            $this->actualizarIndicadoresOtrosEnfoques($ano, $mes);

            $this->actualizarFechaConsolidacion($ano, $mes);
        });
    }

    public function actualizarIndicadoresGlobales($ano, $mes) {
        $dimensiones = array('mun_fk_id', 'rpr_fk_id', 'plv_pk_id');

        $indicadores = "
            count(*) as ipg_visitas_programadas,
            count(*) filter(where plv_realizada = false) as ipg_visitas_por_realizar,
            count(*) filter(where plv_realizada = true) as ipg_visitas_realizadas,
            count(*) filter(where plv_realizada = true and plv_completada = true) as ipg_visitas_realizadas_completadas,
            count(*) filter(where esv_fk_id = 1) as ipg_visitas_efectivas,
            count(*) filter(where esv_fk_id = 2) as ipg_visitas_no_efectivas,
            coalesce(sum(plv_total_preguntas) filter(where plv_completada = true), 0) as ipg_total_criterios,
            coalesce(sum(plv_preguntas_cumplen) filter(where plv_completada = true), 0) as ipg_total_criterios_cumplen,
            coalesce(sum(plv_preguntas_no_cumplen) filter(where plv_completada = true), 0) as ipg_total_criterios_no_cumplen,
            count(*) filter(where plv_completada = true and plv_resultado <= 5) as ipg_visitas_ejec_baja,
            count(*) filter(where plv_completada = true and plv_resultado >= 6 and plv_resultado <= 8) as ipg_visitas_ejec_media,
            count(*) filter(where plv_completada = true and plv_resultado >= 9) as ipg_visitas_ejec_alta,
            count(*) filter(where plv_completada = true and plv_evaluacion_1 = true) as ipg_evaluacion_1_cumple,
            count(*) filter(where plv_completada = true and plv_evaluacion_2 = true) as ipg_evaluacion_2_cumple,
            count(*) filter(where plv_completada = true and plv_evaluacion_3 = true) as ipg_evaluacion_3_cumple,
            count(*) filter(where plv_completada = true and plv_evaluacion_4 = true) as ipg_evaluacion_4_cumple,
            count(*) filter(where plv_completada = true and plv_evaluacion_5 = true) as ipg_evaluacion_5_cumple,
            count(*) filter(where plv_completada = true and plv_evaluacion_6 = true) as ipg_evaluacion_6_cumple,
            count(*) filter(where plv_completada = true and plv_evaluacion_7 = true) as ipg_evaluacion_7_cumple,
            count(*) filter(where plv_completada = true and plv_evaluacion_8 = true) as ipg_evaluacion_8_cumple,
            count(*) filter(where plv_completada = true and plv_evaluacion_9 = true) as ipg_evaluacion_9_cumple,
            count(*) filter(where plv_completada = true and plv_evaluacion_10 = true) as ipg_evaluacion_10_cumple
        ";

        $querySinDimensiones = "
            insert into \"".$this->cliente."\".indicadores_pamec_global (
                ipg_ano, ipg_mes, mun_fk_id, rpr_fk_id, plv_fk_id,
                ipg_visitas_programadas, ipg_visitas_por_realizar, ipg_visitas_realizadas, ipg_visitas_realizadas_completadas, ipg_visitas_efectivas,
                ipg_visitas_no_efectivas, ipg_total_criterios, ipg_total_criterios_cumplen, ipg_total_criterios_no_cumplen, ipg_visitas_ejec_baja,
                ipg_visitas_ejec_media, ipg_visitas_ejec_alta, ipg_evaluacion_1_cumple, ipg_evaluacion_2_cumple, ipg_evaluacion_3_cumple, ipg_evaluacion_4_cumple,
                ipg_evaluacion_5_cumple, ipg_evaluacion_6_cumple, ipg_evaluacion_7_cumple, ipg_evaluacion_8_cumple, ipg_evaluacion_9_cumple, ipg_evaluacion_10_cumple
            )
            select ?::smallint, ?::smallint, -1, -1, -1,
                   @indicadores@
            from \"".$this->cliente."\".plan_visitas_pamec
            where plv_eliminada = false and plv_ano = ? and plv_mes = ?
        ";

        $queryBase = "
                insert into \"".$this->cliente."\".indicadores_pamec_global (
                    ipg_ano, ipg_mes, mun_fk_id, rpr_fk_id, plv_fk_id,
                    ipg_visitas_programadas, ipg_visitas_por_realizar, ipg_visitas_realizadas, ipg_visitas_realizadas_completadas, ipg_visitas_efectivas,
                    ipg_visitas_no_efectivas, ipg_total_criterios, ipg_total_criterios_cumplen, ipg_total_criterios_no_cumplen, ipg_visitas_ejec_baja,
                    ipg_visitas_ejec_media, ipg_visitas_ejec_alta, ipg_evaluacion_1_cumple, ipg_evaluacion_2_cumple, ipg_evaluacion_3_cumple, ipg_evaluacion_4_cumple,
                    ipg_evaluacion_5_cumple, ipg_evaluacion_6_cumple, ipg_evaluacion_7_cumple, ipg_evaluacion_8_cumple, ipg_evaluacion_9_cumple, ipg_evaluacion_10_cumple
                )
                select ?::smallint,
                       ?::smallint,
                       @dimensiones@
                       @indicadores@
                from \"".$this->cliente."\".plan_visitas_pamec as plv
                @join@
                where plv_eliminada = false and plv_ano = ? and plv_mes = ?
                group by 1,2,3,4,5
        ";

        // Limpiar los indicadores del periodo
        DB::connection($this->cliente)->statement("
            delete from \"".$this->cliente."\".indicadores_pamec_global
            where ipg_ano = ? and ipg_mes = ?
        ", [$ano, $mes]);

        // Ejecutar el query sin dimensiones
        $querySDModificado = str_replace("@indicadores@", trim($indicadores), $querySinDimensiones);
        DB::connection($this->cliente)->statement($querySDModificado, [$ano, $mes, $ano, $mes]);

        // Ejecutar el query con dimensiones
        foreach ($dimensiones as $dim) {
            $dimensionesProcesadas = "";
            $joinProcesados = "";

            if ($dim === "mun_fk_id") {
                $dimensionesProcesadas = "mun_fk_id, -1, -1,";
                $joinProcesados = "join base.reps_prestadores on (rpr_pk_id = rpr_fk_id)";
            } else if ($dim === "rpr_fk_id") {
                $dimensionesProcesadas = "-1, rpr_fk_id, -1,";
            } else if ($dim === "plv_pk_id") {
                $dimensionesProcesadas = "-1, -1, plv_pk_id,";
            }

            $queryBaseModificado = str_replace("@dimensiones@", trim($dimensionesProcesadas), $queryBase);
            $queryBaseModificado = str_replace("@join@", trim($joinProcesados), $queryBaseModificado);
            $queryBaseModificado = str_replace("@indicadores@", trim($indicadores), $queryBaseModificado);

            DB::connection($this->cliente)->statement($queryBaseModificado, [$ano, $mes, $ano, $mes]);
        }
    }

    public function actualizarIndicadoresAcreditacion($ano, $mes) {
        $dimensiones = array('mun_fk_id', 'rpr_fk_id', 'plv_pk_id');

        $indicadores = "
            count(*) as ipa_pamec_acreditacion,
            count(*) filter(where plv_resultado <= 5) as ipa_acred_ejec_baja,
            count(*) filter(where plv_resultado >= 6 and plv_resultado <= 8) as ipa_acred_ejec_media,
            count(*) filter(where plv_resultado >= 9) as ipa_acred_ejec_alta,
            count(*) filter(where plv_evaluacion_1 = true) as ipa_acred_evaluacion_1_cumple,
            count(*) filter(where plv_evaluacion_2 = true) as ipa_acred_evaluacion_2_cumple,
            count(*) filter(where plv_evaluacion_3 = true) as ipa_acred_evaluacion_3_cumple,
            count(*) filter(where plv_evaluacion_4 = true) as ipa_acred_evaluacion_4_cumple,
            count(*) filter(where plv_evaluacion_5 = true) as ipa_acred_evaluacion_5_cumple,
            count(*) filter(where plv_evaluacion_6 = true) as ipa_acred_evaluacion_6_cumple,
            count(*) filter(where plv_evaluacion_7 = true) as ipa_acred_evaluacion_7_cumple,
            count(*) filter(where plv_evaluacion_8 = true) as ipa_acred_evaluacion_8_cumple,
            count(*) filter(where plv_evaluacion_9 = true) as ipa_acred_evaluacion_9_cumple,
            count(*) filter(where plv_evaluacion_10 = true) as ipa_acred_evaluacion_10_cumple
        ";

        $querySinDimensiones = "
            insert into \"".$this->cliente."\".indicadores_pamec_acreditacion (
                ipa_ano, ipa_mes, mun_fk_id, rpr_fk_id, plv_fk_id,
                ipa_pamec_acreditacion, ipa_acred_ejec_baja, ipa_acred_ejec_media, ipa_acred_ejec_alta, ipa_acred_evaluacion_1_cumple,
                ipa_acred_evaluacion_2_cumple, ipa_acred_evaluacion_3_cumple, ipa_acred_evaluacion_4_cumple, ipa_acred_evaluacion_5_cumple,
                ipa_acred_evaluacion_6_cumple, ipa_acred_evaluacion_7_cumple, ipa_acred_evaluacion_8_cumple, ipa_acred_evaluacion_9_cumple,
                ipa_acred_evaluacion_10_cumple
            )
            select ?::smallint, ?::smallint, -1, -1, -1,
                   @indicadores@
            from \"".$this->cliente."\".plan_visitas_pamec
            join \"".$this->cliente."\".auditoria_pamec_acreditacion on (plv_fk_id = plv_pk_id)
            where plv_eliminada = false and plv_realizada = true and plv_completada = true and plv_ano = ? and plv_mes = ?
        ";

        $queryBase = "
                insert into \"".$this->cliente."\".indicadores_pamec_acreditacion (
                    ipa_ano, ipa_mes, mun_fk_id, rpr_fk_id, plv_fk_id,
                    ipa_pamec_acreditacion, ipa_acred_ejec_baja, ipa_acred_ejec_media, ipa_acred_ejec_alta, ipa_acred_evaluacion_1_cumple,
                    ipa_acred_evaluacion_2_cumple, ipa_acred_evaluacion_3_cumple, ipa_acred_evaluacion_4_cumple, ipa_acred_evaluacion_5_cumple,
                    ipa_acred_evaluacion_6_cumple, ipa_acred_evaluacion_7_cumple, ipa_acred_evaluacion_8_cumple, ipa_acred_evaluacion_9_cumple,
                    ipa_acred_evaluacion_10_cumple
                )
                select ?::smallint,
                       ?::smallint,
                       @dimensiones@
                       @indicadores@
                from \"".$this->cliente."\".plan_visitas_pamec as plv
                join \"".$this->cliente."\".auditoria_pamec_acreditacion as aup on (plv_fk_id = plv_pk_id)
                @join@
                where plv_eliminada = false and plv_realizada = true and plv_completada = true and plv_ano = ? and plv_mes = ?
                group by 1,2,3,4,5
        ";

        // Limpiar los indicadores del periodo
        DB::connection($this->cliente)->statement("
            delete from \"".$this->cliente."\".indicadores_pamec_acreditacion
            where ipa_ano = ? and ipa_mes = ?
        ", [$ano, $mes]);

        // Ejecutar el query sin dimensiones
        $querySDModificado = str_replace("@indicadores@", trim($indicadores), $querySinDimensiones);
        DB::connection($this->cliente)->statement($querySDModificado, [$ano, $mes, $ano, $mes]);

        // Ejecutar el query con dimensiones
        foreach ($dimensiones as $dim) {
            $dimensionesProcesadas = "";
            $joinProcesados = "";

            if ($dim === "mun_fk_id") {
                $dimensionesProcesadas = "mun_fk_id, -1, -1,";
                $joinProcesados = "join base.reps_prestadores on (rpr_pk_id = plv.rpr_fk_id)";
            } else if ($dim === "rpr_fk_id") {
                $dimensionesProcesadas = "-1, plv.rpr_fk_id, -1,";
            } else if ($dim === "plv_pk_id") {
                $dimensionesProcesadas = "-1, -1, plv_pk_id,";
            }

            $queryBaseModificado = str_replace("@dimensiones@", trim($dimensionesProcesadas), $queryBase);
            $queryBaseModificado = str_replace("@join@", trim($joinProcesados), $queryBaseModificado);
            $queryBaseModificado = str_replace("@indicadores@", trim($indicadores), $queryBaseModificado);

            DB::connection($this->cliente)->statement($queryBaseModificado, [$ano, $mes, $ano, $mes]);
        }
    }

    public function actualizarIndicadoresOtrosEnfoques($ano, $mes) {
        $dimensiones = array('mun_fk_id', 'rpr_fk_id', 'plv_pk_id');

        $indicadores = "
            count(*) filter(where aup_prg_enf_1 = true) as ipo_pamec_si_calidad,
            count(*) filter(where aup_prg_enf_1_calificacion = 1) as ipo_pamec_si_calidad_cumple,
            count(*) filter(where aup_prg_enf_1_calificacion != 1) as ipo_pamec_si_calidad_no_cumple,
            count(*) filter(where aup_prg_enf_1 = true and plv_resultado <= 5) as ipo_si_calidad_ejec_baja,
            count(*) filter(where aup_prg_enf_1 = true and plv_resultado >= 6 and plv_resultado <= 8) as ipo_si_calidad_ejec_media,
            count(*) filter(where aup_prg_enf_1 = true and plv_resultado >= 9) as ipo_si_calidad_ejec_alta,
            count(*) filter(where aup_prg_enf_1 = true and plv_evaluacion_1 = true) as ipo_si_cal_evaluacion_1_cumple,
            count(*) filter(where aup_prg_enf_1 = true and plv_evaluacion_2 = true) as ipo_si_cal_evaluacion_2_cumple,
            count(*) filter(where aup_prg_enf_1 = true and plv_evaluacion_3 = true) as ipo_si_cal_evaluacion_3_cumple,
            count(*) filter(where aup_prg_enf_1 = true and plv_evaluacion_4 = true) as ipo_si_cal_evaluacion_4_cumple,
            count(*) filter(where aup_prg_enf_1 = true and plv_evaluacion_5 = true) as ipo_si_cal_evaluacion_5_cumple,
            count(*) filter(where aup_prg_enf_1 = true and plv_evaluacion_6 = true) as ipo_si_cal_evaluacion_6_cumple,
            count(*) filter(where aup_prg_enf_1 = true and plv_evaluacion_7 = true) as ipo_si_cal_evaluacion_7_cumple,
            count(*) filter(where aup_prg_enf_1 = true and plv_evaluacion_8 = true) as ipo_si_cal_evaluacion_8_cumple,
            count(*) filter(where aup_prg_enf_1 = true and plv_evaluacion_9 = true) as ipo_si_cal_evaluacion_9_cumple,
            count(*) filter(where aup_prg_enf_1 = true and plv_evaluacion_10 = true) as ipo_si_cal_evaluacion_10_cumple,

            count(*) filter(where aup_prg_enf_2 = true) as ipo_pamec_gr_procesos,
            count(*) filter(where aup_prg_enf_2_calificacion = 1) as ipo_pamec_gr_procesos_cumple,
            count(*) filter(where aup_prg_enf_2_calificacion != 1) as ipo_pamec_gr_procesos_no_cumple,
            count(*) filter(where aup_prg_enf_2 = true and plv_resultado <= 5) as ipo_gr_proc_ejec_baja,
            count(*) filter(where aup_prg_enf_2 = true and plv_resultado >= 6 and plv_resultado <= 8) as ipo_gr_proc_ejec_media,
            count(*) filter(where aup_prg_enf_2 = true and plv_resultado >= 9) as ipo_gr_proc_ejec_alta,
            count(*) filter(where aup_prg_enf_2 = true and plv_evaluacion_1 = true) as ipo_gr_proc_evaluacion_1_cumple,
            count(*) filter(where aup_prg_enf_2 = true and plv_evaluacion_2 = true) as ipo_gr_proc_evaluacion_2_cumple,
            count(*) filter(where aup_prg_enf_2 = true and plv_evaluacion_3 = true) as ipo_gr_proc_evaluacion_3_cumple,
            count(*) filter(where aup_prg_enf_2 = true and plv_evaluacion_4 = true) as ipo_gr_proc_evaluacion_4_cumple,
            count(*) filter(where aup_prg_enf_2 = true and plv_evaluacion_5 = true) as ipo_gr_proc_evaluacion_5_cumple,
            count(*) filter(where aup_prg_enf_2 = true and plv_evaluacion_6 = true) as ipo_gr_proc_evaluacion_6_cumple,
            count(*) filter(where aup_prg_enf_2 = true and plv_evaluacion_7 = true) as ipo_gr_proc_evaluacion_7_cumple,
            count(*) filter(where aup_prg_enf_2 = true and plv_evaluacion_8 = true) as ipo_gr_proc_evaluacion_8_cumple,
            count(*) filter(where aup_prg_enf_2 = true and plv_evaluacion_9 = true) as ipo_gr_proc_evaluacion_9_cumple,
            count(*) filter(where aup_prg_enf_2 = true and plv_evaluacion_10 = true) as ipo_gr_proc_evaluacion_10_cumple,

            count(*) filter(where aup_prg_enf_3 = true) as ipo_pamec_seg_paciente,
            count(*) filter(where aup_prg_enf_3_calificacion = 1) as ipo_pamec_seg_paciente_cumple,
            count(*) filter(where aup_prg_enf_3_calificacion != 1) as ipo_pamec_seg_paciente_no_cumple,
            count(*) filter(where aup_prg_enf_3 = true and plv_resultado <= 5) as ipo_seg_pac_ejec_baja,
            count(*) filter(where aup_prg_enf_3 = true and plv_resultado >= 6 and plv_resultado <= 8) as ipo_seg_pac_ejec_media,
            count(*) filter(where aup_prg_enf_3 = true and plv_resultado >= 9) as ipo_seg_pac_ejec_alta,
            count(*) filter(where aup_prg_enf_3 = true and plv_evaluacion_1 = true) as ipo_seg_pac_evaluacion_1_cumple,
            count(*) filter(where aup_prg_enf_3 = true and plv_evaluacion_2 = true) as ipo_seg_pac_evaluacion_2_cumple,
            count(*) filter(where aup_prg_enf_3 = true and plv_evaluacion_3 = true) as ipo_seg_pac_evaluacion_3_cumple,
            count(*) filter(where aup_prg_enf_3 = true and plv_evaluacion_4 = true) as ipo_seg_pac_evaluacion_4_cumple,
            count(*) filter(where aup_prg_enf_3 = true and plv_evaluacion_5 = true) as ipo_seg_pac_evaluacion_5_cumple,
            count(*) filter(where aup_prg_enf_3 = true and plv_evaluacion_6 = true) as ipo_seg_pac_evaluacion_6_cumple,
            count(*) filter(where aup_prg_enf_3 = true and plv_evaluacion_7 = true) as ipo_seg_pac_evaluacion_7_cumple,
            count(*) filter(where aup_prg_enf_3 = true and plv_evaluacion_8 = true) as ipo_seg_pac_evaluacion_8_cumple,
            count(*) filter(where aup_prg_enf_3 = true and plv_evaluacion_9 = true) as ipo_seg_pac_evaluacion_9_cumple,
            count(*) filter(where aup_prg_enf_3 = true and plv_evaluacion_10 = true) as ipo_seg_pac_evaluacion_10_cumple
        ";

        $querySinDimensiones = "
            insert into \"".$this->cliente."\".indicadores_pamec_otros_enfoques (
                ipo_ano, ipo_mes, mun_fk_id, rpr_fk_id, plv_fk_id,
                ipo_pamec_si_calidad, ipo_pamec_si_calidad_cumple, ipo_pamec_si_calidad_no_cumple, ipo_si_calidad_ejec_baja,
                ipo_si_calidad_ejec_media, ipo_si_calidad_ejec_alta, ipo_si_cal_evaluacion_1_cumple, ipo_si_cal_evaluacion_2_cumple,
                ipo_si_cal_evaluacion_3_cumple, ipo_si_cal_evaluacion_4_cumple, ipo_si_cal_evaluacion_5_cumple, ipo_si_cal_evaluacion_6_cumple,
                ipo_si_cal_evaluacion_7_cumple, ipo_si_cal_evaluacion_8_cumple, ipo_si_cal_evaluacion_9_cumple, ipo_si_cal_evaluacion_10_cumple,
                ipo_pamec_gr_procesos, ipo_pamec_gr_procesos_cumple, ipo_pamec_gr_procesos_no_cumple, ipo_gr_proc_ejec_baja,
                ipo_gr_proc_ejec_media, ipo_gr_proc_ejec_alta, ipo_gr_proc_evaluacion_1_cumple, ipo_gr_proc_evaluacion_2_cumple,
                ipo_gr_proc_evaluacion_3_cumple, ipo_gr_proc_evaluacion_4_cumple, ipo_gr_proc_evaluacion_5_cumple, ipo_gr_proc_evaluacion_6_cumple,
                ipo_gr_proc_evaluacion_7_cumple, ipo_gr_proc_evaluacion_8_cumple, ipo_gr_proc_evaluacion_9_cumple, ipo_gr_proc_evaluacion_10_cumple,
                ipo_pamec_seg_paciente, ipo_pamec_seg_paciente_cumple, ipo_pamec_seg_paciente_no_cumple, ipo_seg_pac_ejec_baja,
                ipo_seg_pac_ejec_media, ipo_seg_pac_ejec_alta, ipo_seg_pac_evaluacion_1_cumple, ipo_seg_pac_evaluacion_2_cumple,
                ipo_seg_pac_evaluacion_3_cumple, ipo_seg_pac_evaluacion_4_cumple, ipo_seg_pac_evaluacion_5_cumple, ipo_seg_pac_evaluacion_6_cumple,
                ipo_seg_pac_evaluacion_7_cumple, ipo_seg_pac_evaluacion_8_cumple, ipo_seg_pac_evaluacion_9_cumple, ipo_seg_pac_evaluacion_10_cumple
            )
            select ?::smallint, ?::smallint, -1, -1, -1,
                   @indicadores@
            from \"".$this->cliente."\".plan_visitas_pamec
            join \"".$this->cliente."\".auditoria_pamec_otros on (plv_fk_id = plv_pk_id)
            where plv_eliminada = false and plv_realizada = true and plv_completada = true and plv_ano = ? and plv_mes = ?
        ";

        $queryBase = "
                insert into \"".$this->cliente."\".indicadores_pamec_otros_enfoques (
                    ipo_ano, ipo_mes, mun_fk_id, rpr_fk_id, plv_fk_id,
                    ipo_pamec_si_calidad, ipo_pamec_si_calidad_cumple, ipo_pamec_si_calidad_no_cumple, ipo_si_calidad_ejec_baja,
                    ipo_si_calidad_ejec_media, ipo_si_calidad_ejec_alta, ipo_si_cal_evaluacion_1_cumple, ipo_si_cal_evaluacion_2_cumple,
                    ipo_si_cal_evaluacion_3_cumple, ipo_si_cal_evaluacion_4_cumple, ipo_si_cal_evaluacion_5_cumple, ipo_si_cal_evaluacion_6_cumple,
                    ipo_si_cal_evaluacion_7_cumple, ipo_si_cal_evaluacion_8_cumple, ipo_si_cal_evaluacion_9_cumple, ipo_si_cal_evaluacion_10_cumple,
                    ipo_pamec_gr_procesos, ipo_pamec_gr_procesos_cumple, ipo_pamec_gr_procesos_no_cumple, ipo_gr_proc_ejec_baja,
                    ipo_gr_proc_ejec_media, ipo_gr_proc_ejec_alta, ipo_gr_proc_evaluacion_1_cumple, ipo_gr_proc_evaluacion_2_cumple,
                    ipo_gr_proc_evaluacion_3_cumple, ipo_gr_proc_evaluacion_4_cumple, ipo_gr_proc_evaluacion_5_cumple, ipo_gr_proc_evaluacion_6_cumple,
                    ipo_gr_proc_evaluacion_7_cumple, ipo_gr_proc_evaluacion_8_cumple, ipo_gr_proc_evaluacion_9_cumple, ipo_gr_proc_evaluacion_10_cumple,
                    ipo_pamec_seg_paciente, ipo_pamec_seg_paciente_cumple, ipo_pamec_seg_paciente_no_cumple, ipo_seg_pac_ejec_baja,
                    ipo_seg_pac_ejec_media, ipo_seg_pac_ejec_alta, ipo_seg_pac_evaluacion_1_cumple, ipo_seg_pac_evaluacion_2_cumple,
                    ipo_seg_pac_evaluacion_3_cumple, ipo_seg_pac_evaluacion_4_cumple, ipo_seg_pac_evaluacion_5_cumple, ipo_seg_pac_evaluacion_6_cumple,
                    ipo_seg_pac_evaluacion_7_cumple, ipo_seg_pac_evaluacion_8_cumple, ipo_seg_pac_evaluacion_9_cumple, ipo_seg_pac_evaluacion_10_cumple
                )
                select ?::smallint,
                       ?::smallint,
                       @dimensiones@
                       @indicadores@
                from \"".$this->cliente."\".plan_visitas_pamec as plv
                join \"".$this->cliente."\".auditoria_pamec_otros as aup on (plv_fk_id = plv_pk_id)
                @join@
                where plv_eliminada = false and plv_realizada = true and plv_completada = true and plv_ano = ? and plv_mes = ?
                group by 1,2,3,4,5
        ";

        // Limpiar los indicadores del periodo
        DB::connection($this->cliente)->statement("
            delete from \"".$this->cliente."\".indicadores_pamec_otros_enfoques
            where ipo_ano = ? and ipo_mes = ?
        ", [$ano, $mes]);

        // Ejecutar el query sin dimensiones
        $querySDModificado = str_replace("@indicadores@", trim($indicadores), $querySinDimensiones);
        DB::connection($this->cliente)->statement($querySDModificado, [$ano, $mes, $ano, $mes]);

        // Ejecutar el query con dimensiones
        foreach ($dimensiones as $dim) {
            $dimensionesProcesadas = "";
            $joinProcesados = "";

            if ($dim === "mun_fk_id") {
                $dimensionesProcesadas = "mun_fk_id, -1, -1,";
                $joinProcesados = "join base.reps_prestadores on (rpr_pk_id = plv.rpr_fk_id)";
            } else if ($dim === "rpr_fk_id") {
                $dimensionesProcesadas = "-1, plv.rpr_fk_id, -1,";
            } else if ($dim === "plv_pk_id") {
                $dimensionesProcesadas = "-1, -1, plv_pk_id,";
            }

            $queryBaseModificado = str_replace("@dimensiones@", trim($dimensionesProcesadas), $queryBase);
            $queryBaseModificado = str_replace("@join@", trim($joinProcesados), $queryBaseModificado);
            $queryBaseModificado = str_replace("@indicadores@", trim($indicadores), $queryBaseModificado);

            DB::connection($this->cliente)->statement($queryBaseModificado, [$ano, $mes, $ano, $mes]);
        }
    }

    private function actualizarFechaConsolidacion($ano, $mes) {
        $registro = EstadoConsolidacionIndicadores::on($this->cliente)
                                                  ->where('eci_ano', $ano)
                                                  ->where('eci_mes', $mes)
                                                  ->get()->toArray();

        if (count($registro) > 0) {
            EstadoConsolidacionIndicadores::on($this->cliente)->where('eci_pk_id', $registro[0]['eci_pk_id'])->update([
                'eci_auditoria_pamec' => 'now()'
            ]);
        } else {
            EstadoConsolidacionIndicadores::on($this->cliente)->create([
                'eci_ano' => $ano,
                'eci_mes' => $mes,
                'eci_auditoria_pamec' => 'now()'
            ]);
        }
    }
}
