<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;

use App\Http\Services\DBClientConnection;

use App\Models\clientes\PlanVisitasRias;
use App\Models\clientes\AuditoresVisitasRias;
use App\Models\base\User;
use App\Models\base\RepsPrestadores;
use App\Models\base\RepsPrestadoresSedes;

use App\Http\Controllers\API\MailController;

class PlanVisitasRiasController extends Controller
{
    private $cliente;

    function __construct() {
        // Crear la conexion temporal al esquema del cliente
        $this->middleware(function ($request, $next) {
            if (Auth::user() !== null) {
                $this->cliente = Auth::user()->cli_fk_id;
                $connection = new DBClientConnection();
                config(['database.connections.'.$this->cliente => $connection->getConnectionArray($this->cliente)]);
            }

            return $next($request);
        });
    }

    public function obtenerPlanVisitas(Request $request, $ano, $mes) {
        return PlanVisitasRias::on($this->cliente)
                              ->selectRaw("
                                plan_visitas_rias.*, rpr.*, rps.*, esv,*,
                                (
                                    select count(*) filter (where avr_visita_sincronizada)
                                    from auditores_visitas_rias
                                    where plv_fk_id = plan_visitas_rias.plv_pk_id
                                ) as plv_sincronizada
                              ")
                              ->join('base.reps_prestadores as rpr', 'rpr_fk_id', 'rpr_pk_id')
                              ->join('base.reps_prestadores_sedes as rps', 'rps_fk_id', 'rps_pk_id')
                              ->leftJoin('base.estados_visita as esv', 'esv_fk_id', 'esv_pk_id')
                              ->where('plv_ano', $ano)
                              ->where('plv_mes', $mes)
                              ->where('plv_eliminada', false)
                              ->orderBy('plv_fecha_visita', 'asc')
                              ->get()->toArray();
    }

    public function obtenerAuditoresVisita(Request $request, $idVisita) {
        return AuditoresVisitasRias::on($this->cliente)
                                   ->selectRaw('auditores_visitas_rias.*, tdo.*, usu.name, usu.usu_documento, usu.email, usu.usu_cargo')
                                   ->join('base.users as usu', 'id', 'usu_fk_id')
                                   ->join('base.tipos_documento as tdo', 'tdo_pk_id', 'tdo_fk_id')
                                   ->where('plv_fk_id', $idVisita)
                                   ->get()->toArray();
    }

    public function buscarAsignacionPrestador(Request $request, $ano, $mes, $idPrestador, $idSede) {
        return PlanVisitasRias::on($this->cliente)
                              ->where('plv_ano', $ano)
                              ->where('plv_mes', $mes)
                              ->where('plv_eliminada', false)
                              ->where('rpr_fk_id', $idPrestador)
                              ->where('rps_fk_id', $idSede)
                              ->count();
    }

    public function crearPlanVisitas(Request $request) {
        $id = null;
        $codigoResponsable = strtoupper(Str::random(4));

        DB::connection($this->cliente)->transaction(function() use($request, &$id, $codigoResponsable) {
            $ano = $request->input('ano');
            $mes = $request->input('mes');
            $idPrestador = $request->input('idPrestador');
            $idSede = $request->input('idSede');
            $auditoresVisita = $request->input('auditoresVisita');
            $comentario = $request->input('comentario');

            $id = PlanVisitasRias::on($this->cliente)->create([
                'plv_ano' => $ano,
                'plv_mes' => $mes,
                'rpr_fk_id' => $idPrestador,
                'rps_fk_id' => $idSede,
                'plv_comentario' => $comentario,
                'plv_codigo_visita' => $codigoResponsable,
                'plv_creada_por' => Auth::user()->id
            ])->plv_pk_id;

            $idCoordinador = null;

            foreach ($auditoresVisita as $avr) {
                AuditoresVisitasRias::on($this->cliente)->create([
                    'plv_fk_id' => $id,
                    'avr_ano' => $ano,
                    'avr_mes' => $mes,
                    'usu_fk_id' => $avr['usu_fk_id'],
                    'avr_responsable' => $avr['avr_responsable']
                ]);

                if ($avr['avr_responsable']) {
                    $idCoordinador = $avr['usu_fk_id'];
                }
            }

            // Envío de correo al coordinador
            $correoCoordinador = User::where('id', $idCoordinador)->pluck('email')[0];

            if ($correoCoordinador !== null) {
                $nombrePrestador = RepsPrestadores::where('rpr_pk_id', $idPrestador)->pluck('rpr_razon_social')[0];
                $nombreSede = RepsPrestadoresSedes::where('rps_pk_id', $idSede)->pluck('rps_nombre_sede')[0];

                (new MailController)->enviarNotificacionVisitaRias($correoCoordinador, $nombrePrestador, $nombreSede, $codigoResponsable);
            }
        });

        return [$id, $codigoResponsable];
    }

    public function actualizarPlanVisitas(Request $request, $idVisita) {
        DB::connection($this->cliente)->transaction(function() use($request, &$idVisita) {
            $auditoresVisita = $request->input('auditoresVisita');
            $comentario = $request->input('comentario');

            PlanVisitasRias::on($this->cliente)->where('plv_pk_id', $idVisita)->update(['plv_comentario' => $comentario]);

            foreach ($auditoresVisita as $avr) {
                if (isset($avr['avr_borrado']) && $avr['avr_borrado']) {
                    AuditoresVisitasRias::on($this->cliente)->where('avr_pk_id', $avr['avr_pk_id'])->delete();
                }
            }
        });
    }

    public function eliminarPrestadorPlanVisitas(Request $request, $idVisita) {
        PlanVisitasRias::on($this->cliente)->where('plv_pk_id', $idVisita)
        ->update([
            'plv_eliminada' => true,
            'plv_borrada_por' => Auth::user()->id
        ]);
    }

    public function actualizarFechaVisita(Request $request, $idVisita) {
        PlanVisitasRias::on($this->cliente)
                       ->where('plv_pk_id', $idVisita)
                       ->update(['plv_fecha_visita' => $request->input('plv_fecha_visita')]);
    }

    public function actualizarSincronizacionVisita(Request $request, $idVisita) {
        $estado = $request->input('estado');

        AuditoresVisitasRias::on($this->cliente)
                            ->where('plv_fk_id', $idVisita)
                            ->where('usu_fk_id', Auth::user()->id)
                            ->update(['avr_visita_sincronizada' => $estado]);
    }


    /*
    public function finalizarVisita(Request $request, $idVisita) {
        DB::transaction(function () use ($request, $idVisita) {
            PlanVisitasPamec::on($this->cliente)->where('plv_pk_id', $idVisita)
            ->update([
                'plv_realizada' => true,
                'plv_fecha_cierre_visita' => 'now()',
                'esv_fk_id' => $request->input('esv_fk_id'),
                'plv_comentario_cierre' => $request->input('plv_comentario_cierre')
            ]);
        });
    }*/
}
