<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;
use Illuminate\Support\Carbon;

use App\Models\base\User;
use App\Models\hospitalia\HospitaliaUser;
use App\Models\clientes\Licencias;
use App\Models\clientes\ParametrosVerifica;
use App\Models\hospitalia\Clientes;
use App\Models\base\OAuthAccessTokens;

use GuzzleHttp\Client;
use GuzzleHttp\Exception\RequestException;
use App\Http\Services\DBClientConnection;
use Illuminate\Support\Facades\Log;

class UsuariosController extends Controller
{
    public function autenticarUsuario(Request $request) {
        $response = null;

        DB::transaction(function() use ($request, &$response) {
            $tokenValido = false;
            $token = $request->token;

            $infoUsuario = null;

            // Validar que el token de Hospitalia sea correcto
            try {
                $httpClient = new Client(['headers' => ['Authorization' => $token]]);
                $response = $httpClient->get(env('HSP_API_URL').'/api/v1/acceso/validarToken');
                $infoUsuario = json_decode($response->getBody(), true);
                $tokenValido = true;
            } catch (RequestException $e) {
                Log::error('LOGIN FAILED: HOSPITALIA TOKEN NOT VALID', [
                    'message' => $e->getMessage(),
                    'request' => $e->getRequest()->getMethod() . ' ' . $e->getRequest()->getUri(),
                    'response' => $e->hasResponse() ? $e->getResponse()->getStatusCode() . ' ' . $e->getResponse()->getBody()->getContents() : 'No response',
                ]);
            }

            if ($tokenValido) {
                // Obtener la información del cliente
                $cliente = Clientes::on('hospitalia')
                                    ->selectRaw('clientes.*,
                                                coalesce(least(
                                                    cli_mod_verifica_fecha_inicio, cli_mod_synriesgo_fecha_inicio, cli_mod_evaluar_fecha_inicio,
                                                    cli_mod_eadversos_fecha_inicio, cli_mod_costos_hosp_fecha_inicio
                                                ), \'2090-10-22\'::date) as cli_fecha_inicio,
                                                coalesce(greatest(
                                                    cli_mod_verifica_fecha_fin, cli_mod_synriesgo_fecha_fin, cli_mod_evaluar_fecha_fin,
                                                    cli_mod_eadversos_fecha_fin, cli_mod_costos_hosp_fecha_fin
                                                ), \'1995-10-22\'::date) as cli_fecha_fin,
                                                cli_mod_verifica_fecha_inicio as cli_fecha_inicio_verifica,
                                                cli_mod_verifica_fecha_fin as cli_fecha_fin_verifica')
                                    ->where('cli_pk_id', $infoUsuario['cli_fk_id'])
                                    ->get()->toArray()[0];

                $fechasContrato = $this->calcularValidezFechas($cliente['cli_fecha_inicio'], $cliente['cli_fecha_fin']);
                $fechasVerifica = $this->calcularValidezFechas($cliente['cli_fecha_inicio_verifica'], $cliente['cli_fecha_fin_verifica']);

                if ($cliente['cli_activo'] && $cliente['cli_mod_verifica'] && $fechasContrato && $fechasVerifica) {
                    // Validar que el usuario exista en Verifica
                    $infoVerificaUsuario = User::where('id', $infoUsuario['id'])->first();

                    if ($infoVerificaUsuario) {
                        // Validar que el rol tenga permiso de acceso en la versión web
                        if (intval($infoVerificaUsuario['rol_fk_id']) <= 3) {
                            // Validar que el usuario tenga licencia en Verifica
                            $cliente = $infoUsuario['cli_fk_id'];
                            $connection = new DBClientConnection();
                            config(['database.connections.'.$cliente => $connection->getConnectionArray(strval($cliente))]);

                            $licencia = Licencias::on($cliente)->where('usu_fk_id', $infoUsuario['id'])->count();

                            if ($licencia > 0) {
                                $scope = $this->obtenerScopePorRol($infoVerificaUsuario->rol_fk_id);

                                // Borrar tokens de acceso anteriores
                                $this->borrarTokensAcceso($infoVerificaUsuario, $scope.'-'.$infoUsuario['email']);

                                // Crear nuevo token
                                $token = $this->encriptarTokenAcceso($this->crearTokenAcceso($infoVerificaUsuario, $infoUsuario['email'], $scope));

                                $almacenamientoNube = Clientes::on('hospitalia')->where('cli_pk_id', $cliente)->pluck('cli_almacenamiento_nube')[0];
                                $parametros = ParametrosVerifica::on($cliente)->get()->toArray()[0];

                                $response = response()->json([
                                    'estado' => true,
                                    'id' => $infoVerificaUsuario->id,
                                    'token' => $token,
                                    'usuario' => [
                                        'rol_fk_id' => $infoVerificaUsuario->rol_fk_id,
                                        'name' => $infoUsuario['name'],
                                        'email' => $infoUsuario['email'],
                                        'tdo_fk_id' => $infoUsuario['tdo_fk_id'],
                                        'usu_documento' => $infoUsuario['usu_documento'],
                                        'dep_fk_id' => $parametros['pav_departamento'],
                                        'mun_fk_id' => $parametros['pav_municipio'],
                                        'cli_almacenamiento_nube' => $almacenamientoNube,
                                        'usu_cargo' => $infoVerificaUsuario->usu_cargo,
                                        'usu_mod_suh_3100' => $infoVerificaUsuario->usu_mod_suh_3100,
                                        'usu_mod_aud_calidad' => $infoVerificaUsuario->usu_mod_aud_calidad,
                                        'usu_mod_rias' => $infoVerificaUsuario->usu_mod_rias,
                                        'usu_mod_pamec' => $infoVerificaUsuario->usu_mod_pamec
                                    ]
                                ], 200);
                            } else {
                                $response = response()->json(['estado' => false, 'error' => 'Usted no tiene una licencia asignada en Verifica'], 200);
                            }
                        } else {
                            $response = response()->json(['estado' => false, 'error' => 'Por el rol asignado, usted debe acceder a Verifica desde la aplicación móvil'], 200);
                        }
                    } else {
                        $response = response()->json(['estado' => false, 'error' => 'Su usuario no está registrado en Verifica. Contacte al responsable'], 200);
                    }
                } else {
                    $response = response()->json(['estado' => false, 'error' => 'Se ha finalizado la licencia para esté módulo. Contacte al responsable'], 200);
                }
            } else {
                $response = response()->json(['estado' => false, 'error' => 'Las credenciales suminsitradas por Hospitalia no son válidas'], 200);
            }
        });

        return $response;
    }

    public function autenticarUsuarioAplicacionMovil(Request $request) {
        $response = null;

        DB::transaction(function() use ($request, &$response) {
            // Obtener el usuario
            $usuarioHospitalia = HospitaliaUser::on('hospitalia')
                                               ->where(['email' => $request->email])
                                               ->where(['usu_activo' => true])
                                               ->where('id', '>', 2)
                                               ->first();

            // Validar que exista el usuario en Hospitalia
            if ($usuarioHospitalia) {
                // Validar que la cuenta esté verificada
                if ($usuarioHospitalia['email_verified_at'] !== null) {
                    // Validar la contraseña
                    if (Hash::check($request->password, $usuarioHospitalia->password)) {
                        // Obtener la información del cliente
                        $cliente = Clientes::on('hospitalia')
                                           ->selectRaw('clientes.*,
                                                        coalesce(least(
                                                            cli_mod_verifica_fecha_inicio, cli_mod_synriesgo_fecha_inicio, cli_mod_evaluar_fecha_inicio,
                                                            cli_mod_eadversos_fecha_inicio, cli_mod_costos_hosp_fecha_inicio
                                                        ), \'2090-10-22\'::date) as cli_fecha_inicio,
                                                        coalesce(greatest(
                                                            cli_mod_verifica_fecha_fin, cli_mod_synriesgo_fecha_fin, cli_mod_evaluar_fecha_fin,
                                                            cli_mod_eadversos_fecha_fin, cli_mod_costos_hosp_fecha_fin
                                                        ), \'1995-10-22\'::date) as cli_fecha_fin,
                                                        cli_mod_verifica_fecha_inicio as cli_fecha_inicio_verifica,
                                                        cli_mod_verifica_fecha_fin as cli_fecha_fin_verifica')
                                           ->where('cli_pk_id', $usuarioHospitalia['cli_fk_id'])
                                           ->get()->toArray()[0];

                        $fechasContrato = $this->calcularValidezFechas($cliente['cli_fecha_inicio'], $cliente['cli_fecha_fin']);
                        $fechasVerifica = $this->calcularValidezFechas($cliente['cli_fecha_inicio_verifica'], $cliente['cli_fecha_fin_verifica']);

                        if ($fechasContrato && $fechasVerifica) {
                            // Validar que el usuario exista en Verifica
                            $usuarioVerifica = User::where('id', $usuarioHospitalia['id'])->first();

                            if ($usuarioVerifica) {
                                // Validar que el rol tenga permiso de acceso en la aplicación
                                if (intval($usuarioVerifica['rol_fk_id']) === 4) {
                                    // Validar que el usuario tenga licencia en Verifica
                                    $idCliente = $usuarioHospitalia['cli_fk_id'];
                                    $connection = new DBClientConnection();
                                    config(['database.connections.'.$idCliente => $connection->getConnectionArray(strval($idCliente))]);

                                    $licencia = Licencias::on($idCliente)->where('usu_fk_id', $usuarioHospitalia['id'])->count();

                                    if ($licencia > 0) {
                                        if ($cliente['cli_doble_factor_auth'] !== null && $cliente['cli_doble_factor_auth']) {
                                            $verificacion = random_int(100000, 999999);

                                            User::where('id', $usuarioHospitalia['id'])->update(['usu_verificacion' => $verificacion]);
                                            (new MailController)->enviarCodigoVerificacion($request->input('email'), $verificacion);

                                            $response = response()->json(['estado' => true, 'id' => $usuarioHospitalia['id'], '2fa' => true], 200);
                                        } else {
                                            $scope = $this->obtenerScopePorRol($usuarioVerifica['rol_fk_id'], true);

                                            // Borrar tokens de acceso anteriores
                                            $this->borrarTokensAcceso($usuarioVerifica, $scope.'-'.$usuarioHospitalia['email']);

                                            // Crear nuevo token
                                            $token = $this->encriptarTokenAcceso($this->crearTokenAcceso($usuarioVerifica, $usuarioHospitalia['email'], $scope));

                                            $almacenamientoNube = Clientes::on('hospitalia')->where('cli_pk_id', $idCliente)->pluck('cli_almacenamiento_nube')[0];
                                            $parametros = ParametrosVerifica::on($idCliente)->get()->toArray()[0];

                                            $response = response()->json([
                                                'estado' => true,
                                                'id' => $usuarioVerifica['id'],
                                                'token' => $token,
                                                'usuario' => [
                                                    'rol_fk_id' => $usuarioVerifica['rol_fk_id'],
                                                    'name' => $usuarioHospitalia['name'],
                                                    'email' => $usuarioHospitalia['email'],
                                                    'tdo_fk_id' => $usuarioHospitalia['tdo_fk_id'],
                                                    'usu_documento' => $usuarioHospitalia['usu_documento'],
                                                    'dep_fk_id' => $parametros['pav_departamento'],
                                                    'mun_fk_id' => $parametros['pav_municipio'],
                                                    'cli_almacenamiento_nube' => $almacenamientoNube,
                                                    'usu_cargo' => $usuarioVerifica['usu_cargo'],
                                                    'usu_mod_suh_3100' => $usuarioVerifica['usu_mod_suh_3100'],
                                                    'usu_mod_aud_calidad' => $usuarioVerifica['usu_mod_aud_calidad'],
                                                    'usu_mod_rias' => $usuarioVerifica['usu_mod_rias'],
                                                    'usu_mod_pamec' => $usuarioVerifica['usu_mod_pamec']
                                                ]
                                            ], 200);
                                        }
                                    } else {
                                        $response = response()->json(['estado' => false, 'error' => 'Usted no tiene una licencia asignada en Verifica'], 200);
                                    }
                                } else {
                                    $response = response()->json(['estado' => false, 'error' => 'Por el rol asignado, usted debe acceder a Verifica desde versión web'], 200);
                                }
                            } else {
                                $response = response()->json(['estado' => false, 'error' => 'Su usuario no está registrado en Verifica. Contacte al responsable'], 200);
                            }
                        } else {
                            $response = response()->json(['estado' => false, 'error' => 'El periodo de acceso a la herramienta ha finalizado. Por favor póngase en contacto con el administrador'], 200);
                        }
                    } else {
                        $response = response()->json(['estado' => false, 'error' => 'Las credenciales suministradas no son correctas'], 200);
                    }
                } else {
                    $response = response()->json(['estado' => false, 'error' => 'No ha validado su cuenta. Por favor revise su correo electrónico'], 200);
                }
            } else {
                $response = response()->json(['estado' => false, 'error' => 'Las credenciales suministradas no son correctas o su usuario no está activo'], 200);
            }
        });

        return $response;
    }

    public function verificarCodigoAccesoMovil(Request $request) {
        $response = null;

        DB::transaction(function() use ($request, &$response) {
            $usuarioVerifica = User::where('id', $request->id)
                                   ->where('usu_verificacion', $request->codigo)
                                   ->first();

            if ($usuarioVerifica) {
                User::where('id', $request->id)->update([
                    'usu_verificacion' => null,
                    'usu_intentos_verificacion' => 0
                ]);

                $usuarioHospitalia = HospitaliaUser::on('hospitalia')->where('id', $request->id)->first();

                $token = $this->encriptarTokenAcceso($this->crearTokenAcceso($usuarioVerifica, $usuarioHospitalia->email, 'ver-aud-mob'));

                $idCliente = $usuarioVerifica->cli_fk_id;
                $connection = new DBClientConnection();
                config(['database.connections.'.$idCliente => $connection->getConnectionArray(strval($idCliente))]);

                $almacenamientoNube = Clientes::on('hospitalia')->where('cli_pk_id', $idCliente)->pluck('cli_almacenamiento_nube')[0];
                $parametros = ParametrosVerifica::on($idCliente)->get()->toArray()[0];

                $response = response()->json([
                    'estado' => true,
                    'id' => $usuarioVerifica['id'],
                    'token' => $token,
                    'usuario' => [
                        'rol_fk_id' => $usuarioVerifica['rol_fk_id'],
                        'name' => $usuarioHospitalia['name'],
                        'email' => $usuarioHospitalia['email'],
                        'tdo_fk_id' => $usuarioHospitalia['tdo_fk_id'],
                        'usu_documento' => $usuarioHospitalia['usu_documento'],
                        'dep_fk_id' => $parametros['pav_departamento'],
                        'mun_fk_id' => $parametros['pav_municipio'],
                        'cli_almacenamiento_nube' => $almacenamientoNube,
                        'usu_cargo' => $usuarioVerifica['usu_cargo'],
                        'usu_mod_suh_3100' => $usuarioVerifica['usu_mod_suh_3100'],
                        'usu_mod_aud_calidad' => $usuarioVerifica['usu_mod_aud_calidad'],
                        'usu_mod_rias' => $usuarioVerifica['usu_mod_rias'],
                        'usu_mod_pamec' => $usuarioVerifica['usu_mod_pamec']
                    ]
                ], 200);
            } else {
                $usuario = User::where('id', $request->id)->first();

                $correo = $usuario->email;
                $usuario->increment('usu_intentos_verificacion');
                $intentos = $usuario->usu_intentos_verificacion;

                if ($intentos >= 3) {
                    User::where('id', $request->id)->update(['usu_verificacion' => null, 'usu_activo' => false]);
                    (new MailController)->notificarBloqueoCuenta($correo);
                }

                $response = response()->json(['estado' => false]);
            }
        });

        return $response;
    }

    public function obtenerRolUsuario(Request $request) {
        return Auth::user()->rol_fk_id;
    }

    public function obtenerUsuariosLicenciados(Request $request) {
        $cliente = Auth::user()->cli_fk_id;

        // Crear la conexión temporal
        $connection = new DBClientConnection();
        config(['database.connections.'.$cliente => $connection->getConnectionArray($cliente)]);

        // Obtener usuarios licenciados
        $usuariosLicenciados = DB::connection($cliente)->select('
            select lic.usu_fk_id as id, usu.rol_fk_id, rol_descripcion, usu.usu_cargo, usu.usu_mod_suh_3100,
                   usu.usu_mod_aud_calidad, usu.usu_mod_rias, usu.usu_mod_pamec, lic_tipo_licencia
            from licencias as lic
            left join "base".users as usu on id = usu_fk_id
            left join "base".roles on rol_pk_id = rol_fk_id
        ');

        // Obtener info usuarios hospitalia
        $idUsuarios = [];

        foreach ($usuariosLicenciados as $usu) {
            array_push($idUsuarios, $usu->id);
        }

        $placeholders = implode(',', array_fill(0, count($idUsuarios), '?'));

        $infoUsuariosHospitalia = DB::connection('hospitalia')->select("
            select id, name, email, tdo_fk_id, tdo_descripcion_corta, usu_documento, usu_activo
            from users
            join tipos_documento on tdo_pk_id = tdo_fk_id
            where id in ($placeholders)
            order by 1
        ", $idUsuarios);

        // Completar la información
        foreach ($usuariosLicenciados as $key => $usu) {
            foreach ($infoUsuariosHospitalia as $usuHosp) {
                if (intval($usuHosp->id) === intval($usu->id)) {
                    $usuariosLicenciados[$key]->name = $usuHosp->name;
                    $usuariosLicenciados[$key]->email = $usuHosp->email;
                    $usuariosLicenciados[$key]->tdo_fk_id = $usuHosp->tdo_fk_id;
                    $usuariosLicenciados[$key]->tdo_descripcion_corta = $usuHosp->tdo_descripcion_corta;
                    $usuariosLicenciados[$key]->usu_documento = $usuHosp->usu_documento;
                    $usuariosLicenciados[$key]->usu_activo = $usuHosp->usu_activo;

                    $usuariosLicenciados[$key]->usu_licenciado = $usu->rol_fk_id !== null;
                    break;
                }
            }
        }

        return $usuariosLicenciados;
    }

    public function obtenerUsuariosRegistrados(Request $request) {
        $cliente = Auth::user()->cli_fk_id;

        DB::connection($cliente)->select('
            select lic.usu_fk_id, lic_tipo_licencia, usu.id, usu.rol_fk_id, usu.usu_cargo, usu.usu_mod_suh_3100,
                usu.usu_mod_aud_calidad, usu.usu_mod_rias, usu.usu_mod_pamec
            from licencias as lic
            left join "base".users as usu on id = usu_fk_id
        ');
    }

    public function asignarLicenciaUsuario(Request $request) {
        return User::create([
            'id' => $request->id,
            'cli_fk_id' => Auth::user()->cli_fk_id,
            'rol_fk_id' => $request->rol_fk_id,
            'usu_cargo' => $request->usu_cargo,
            'usu_mod_aud_calidad' => $request->usu_mod_aud_calidad === null ? false : $request->usu_mod_aud_calidad,
            'usu_mod_pamec' => $request->usu_mod_pamec === null ? false : $request->usu_mod_pamec,
            'usu_mod_rias' => $request->usu_mod_rias === null ? false : $request->usu_mod_rias,
            'usu_mod_suh_3100' => $request->usu_mod_suh_3100 === null ? false : $request->usu_mod_suh_3100
        ]);
    }

    public function actualizarLicenciaUsuario(Request $request, $idUsuario) {
        User::where('id', $idUsuario)->update([
            'rol_fk_id' => $request->rol_fk_id,
            'usu_cargo' => $request->usu_cargo,
            'usu_mod_aud_calidad' => $request->usu_mod_aud_calidad === null ? false : $request->usu_mod_aud_calidad,
            'usu_mod_pamec' => $request->usu_mod_pamec === null ? false : $request->usu_mod_pamec,
            'usu_mod_rias' => $request->usu_mod_rias === null ? false : $request->usu_mod_rias,
            'usu_mod_suh_3100' => $request->usu_mod_suh_3100 === null ? false : $request->usu_mod_suh_3100
        ]);
    }

    public function obtenerUsuariosAuditoresVerificadores(Request $request) {
        $cliente = Auth::user()->cli_fk_id;

        $usuariosVerifica = User::where('cli_fk_id', $cliente)
                                ->selectRaw('id, usu_cargo')
                                ->where('rol_fk_id', 4)
                                ->get()->toArray();

        // Obtener info usuarios hospitalia
        $idUsuarios = [];

        foreach ($usuariosVerifica as $usu) {
            array_push($idUsuarios, $usu['id']);
        }

        $placeholders = implode(',', array_fill(0, count($idUsuarios), '?'));

        $infoUsuariosHospitalia = DB::connection('hospitalia')->select("
            select id, name, email, tdo_fk_id, tdo_descripcion_corta, usu_documento, usu_activo
            from users
            join tipos_documento on tdo_pk_id = tdo_fk_id
            where id in ($placeholders)
            order by 1
        ", $idUsuarios);

        // Completar la información
        foreach ($usuariosVerifica as $key => $usu) {
            foreach ($infoUsuariosHospitalia as $usuHosp) {
                if (intval($usuHosp->id) === intval($usu['id'])) {
                    $usuariosVerifica[$key]['name'] = $usuHosp->name;
                    $usuariosVerifica[$key]['email'] = $usuHosp->email;
                    $usuariosVerifica[$key]['tdo_fk_id'] = $usuHosp->tdo_fk_id;
                    $usuariosVerifica[$key]['tdo_descripcion_corta'] = $usuHosp->tdo_descripcion_corta;
                    $usuariosVerifica[$key]['usu_documento'] = $usuHosp->usu_documento;
                    $usuariosVerifica[$key]['usu_activo'] = $usuHosp->usu_activo;
                    break;
                }
            }
        }

        // Solo dejar los usuarios activos
        $usuariosVerifica = array_filter($usuariosVerifica, function($user) {
            return $user['usu_activo'];
        });

        return $usuariosVerifica;
    }

    public function validarTokenUsuario(Request $request) {
        $usuario = Auth::user();
        $cliente = $usuario->cli_fk_id;
        $tokenHospitalia = $request->auth;

        $tokenHospitaliaCorrecto = false;

        // Validar que el token de Hospitalia sea correcto
        try {
            $httpClient = new Client(['headers' => ['Authorization' => $tokenHospitalia]]);
            $response = $httpClient->get(env('HSP_API_URL').'/api/v1/acceso/validarToken');

            if (intval($response->getStatusCode()) === 200) {
                $tokenHospitaliaCorrecto = true;
            }
        } catch (RequestException $e) {
            if (intval($e->getResponse()->getStatusCode()) !== 401 && intval($e->getResponse()->getStatusCode()) !== 403) {
                $tokenHospitaliaCorrecto = true;
            }

            Log::error('LOGIN FAILED: HOSPITALIA TOKEN NOT VALID', [
                'message' => $e->getMessage(),
                'request' => $e->getRequest()->getMethod() . ' ' . $e->getRequest()->getUri(),
                'response' => $e->hasResponse() ? $e->getResponse()->getStatusCode() . ' ' . $e->getResponse()->getBody()->getContents() : 'No response',
            ]);
        }

        if ($tokenHospitaliaCorrecto) {
            $accesoModulo = Clientes::on('hospitalia')
                                    ->selectRaw('cli_activo,
                                                cli_mod_verifica,
                                                coalesce(least(
                                                    cli_mod_verifica_fecha_inicio, cli_mod_synriesgo_fecha_inicio, cli_mod_evaluar_fecha_inicio,
                                                    cli_mod_eadversos_fecha_inicio, cli_mod_costos_hosp_fecha_inicio
                                                ), \'2090-10-22\'::date) as cli_fecha_inicio,
                                                coalesce(greatest(
                                                    cli_mod_verifica_fecha_fin, cli_mod_synriesgo_fecha_fin, cli_mod_evaluar_fecha_fin,
                                                    cli_mod_eadversos_fecha_fin, cli_mod_costos_hosp_fecha_fin
                                                ), \'1995-10-22\'::date) as cli_fecha_fin,
                                                cli_mod_verifica_fecha_inicio,
                                                cli_mod_verifica_fecha_fin')
                                    ->where('cli_pk_id', $cliente)
                                    ->get()->toArray()[0];

            $fechasContrato = $this->calcularValidezFechas($accesoModulo['cli_fecha_inicio'], $accesoModulo['cli_fecha_fin']);
            $fechasVerifica = $this->calcularValidezFechas($accesoModulo['cli_mod_verifica_fecha_inicio'], $accesoModulo['cli_mod_verifica_fecha_fin']);

            if ($accesoModulo['cli_activo'] && $accesoModulo['cli_mod_verifica'] && $fechasContrato && $fechasVerifica) {
                // Crear la conexión temporal
                $connection = new DBClientConnection();
                config(['database.connections.'.$cliente => $connection->getConnectionArray($cliente)]);

                // Validar que aún tenga licencia
                if (Licencias::on($cliente)->where('usu_fk_id', $usuario->id)->count() > 0) {
                    $almacenamiento = Clientes::on('hospitalia')->where('cli_pk_id', $cliente)->pluck('cli_almacenamiento_nube')[0];

                    return response([
                        'rol_fk_id' => Auth::user()->rol_fk_id,
                        'usuario' => [
                            'cli_almacenamiento_nube' => $almacenamiento,
                            'usu_mod_suh_3100' => $usuario->usu_mod_suh_3100,
                            'usu_mod_aud_calidad' => $usuario->usu_mod_aud_calidad,
                            'usu_mod_rias' => $usuario->usu_mod_rias,
                            'usu_mod_pamec' => $usuario->usu_mod_pamec
                        ]
                    ], 200);
                }
            } else {
                return response(['Licencia vencida'], 401);
            }
        } else {
            return response(['Licencia revocada'], 401);
        }
    }

    public function obtenerEstadoPlazoCambioCodigo(Request $request) {
        $idUsuario = Auth::user()->id;
        $fechaPlazo = User::where('id', $idUsuario)->pluck('usu_plazo_cambio_codigo_bloqueo')[0];

        if ($fechaPlazo !== null) {
            $fechaActual = Carbon::now();
            $fechaPlazo = Carbon::parse($fechaPlazo);

            if (!$fechaActual->greaterThan($fechaPlazo)) {
                return response(['Plazo activo'], 200);
            }

            return response(['El tiempo de actualizacion ha expirado'], 403);
        }

        return response(['El tiempo de actualizacion ha expirado'], 403);
    }

    public function enviarAutorizacionCambioCodigo(Request $request) {
        DB::transaction(function() {
            $infoUsuario = HospitaliaUser::on('hospitalia')
                                         ->select('id', 'email')
                                         ->where('id', Auth::user()->id)
                                         ->get()->toArray()[0];

            $codigoAutorizacion = Str::random(20);

            User::where('id', $infoUsuario['id'])->update([
                'usu_autorizacion_codigo_bloqueo' => $codigoAutorizacion,
                'usu_plazo_cambio_codigo_bloqueo' => null
            ]);

            (new MailController)->enviarAutorizacionCambioBloqueo($infoUsuario['email'], $codigoAutorizacion);
        });
    }

    public function actualizarCodigoBloqueo(Request $request) {
        $idUsuario = Auth::user()->id;

        $fechaActual = Carbon::now();
        $fechaPlazo = Carbon::parse(User::where('id', $idUsuario)->pluck('usu_plazo_cambio_codigo_bloqueo')[0]);

        if (!$fechaActual->greaterThan($fechaPlazo)) {
            User::where('id', $idUsuario)->update([
                'usu_codigo_bloqueo' => Crypt::encryptString($request->codigoBloqueo),
                'usu_autorizacion_codigo_bloqueo' => null,
                'usu_plazo_cambio_codigo_bloqueo' => null
            ]);

            return response(['Codigo actualizado'], 200);
        }

        User::where('id', $idUsuario)->update([
            'usu_autorizacion_codigo_bloqueo' => null,
            'usu_plazo_cambio_codigo_bloqueo' => null
        ]);

        return response(['El tiempo de actualizacion ha expirado'], 403);
    }

    public function cerrarSesion(Request $request) {
        $usuario = Auth::user();
        $scope = $this->obtenerScopePorRol($usuario->rol_fk_id);

        $correo = DB::connection('hospitalia')->select('
            select email
            from users
            where id = ?
        ', [$usuario->id])[0]->email;

        $this->borrarTokensAcceso($usuario, $scope.'-'.$correo);
    }

    private function borrarTokensAcceso($usuario, $nombreToken) {
        $tokens = (array) json_decode(json_encode($usuario->tokens));

        for ($i = 0; $i < count($tokens); $i++) {
            if ($nombreToken === $tokens[$i]->name) {
                OAuthAccessTokens::where('id', $tokens[$i]->id)->delete();
            }
        }

        return response(['Tokens purgados'], 200);
    }

    private function crearTokenAcceso($usuario, $email, $scope = 'pub') {
        return $usuario->createToken($scope.'-'.$email, [$scope])->accessToken;
    }

    private function obtenerScopePorRol($rol, $movil = false) {
        $scope = "";

        if (intval($rol) === 1) {
            $scope = "ver-adm";
        } else if (intval($rol) === 2) {
            $scope = "ver-adm-lid";
        } else if (intval($rol) === 3) {
            $scope = "ver-lid";
        } else if (intval($rol) === 4) {
            if (!$movil) {
                $scope = "ver-aud";
            } else {
                $scope = "ver-aud-mob";
            }
        }

        return $scope;
    }

    private function encriptarTokenAcceso($token) {
        return Crypt::encryptString($token);
    }

    private function calcularValidezFechas($fechaInicio, $fechaFin) {
        if ($fechaInicio === null || $fechaFin === null) {
            return false;
        }

        $fechaActual = new \DateTime();
        $fechaActual->setTime(0, 0, 0);

        $fechaInicio = \DateTime::createFromFormat('Y-m-d', $fechaInicio);
        $fechaInicio->setTime(0, 0, 0);
        $fechaFin = \DateTime::createFromFormat('Y-m-d', $fechaFin);
        $fechaFin->setTime(0, 0, 0);

        if ($fechaActual >= $fechaInicio && $fechaActual <= $fechaFin) {
            return true;
        }

        return false;
    }
}
